/* ----------------------------------------------------------------------------
 * Copyright (c) 2020-2030 Boling Limited. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of Boling nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * -------------------------------------------------------------------------- */

/**
 * @file     app_shell.c
 * @date     30. Jun. 2023
 * @author   Boling SW Team
 *
 * @version
 * Version 1.0
 *  - Initial release
 *
 * @{
 */

/*******************************************************************************
 * INCLUDES
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "shell.h"
#include "omble.h"
#include "om_driver.h"
#include "app_common.h"

/*********************************************************************
 * MACROS
 */

/*********************************************************************
 * LOCAL VARIABLES
 */
const char *busy_flag;

static void cmd_shell_adv(int argc, char *argv[])
{
    if (busy_flag) {
        OM_LOG_DEBUG("%s\n", busy_flag);
        return;
    }
    if (argc <= 0) {
        OM_LOG_DEBUG("Usage(adv): adv start/stop [adv_index=0]\n");
        OM_LOG_DEBUG("\tindex=0: Connectable, others non-connectable\n");
        return;
    }
    uint32_t res;
    int index = argc >= 2 ? atoi(argv[1]) : 0;
    if (!strcmp(argv[0], "start")) {
        res = app_adv_start(index, OB_ADV_PROP_LEGACY_IND);
        OM_LOG_DEBUG("Starting advertise, index=%d, status=%d\n", index, res);
    } else if (!strcmp(argv[0], "stop")) {
        res = app_adv_stop(index);
        OM_LOG_DEBUG("Stopping advertise, index=%d, status=%d\n", index, res);
    } else {
        OM_LOG_DEBUG("Usage(adv): adv start/stop [adv_index=0]\n");
        return;
    }
}

static void cmd_shell_scan(int argc, char *argv[])
{
    if (busy_flag) {
        OM_LOG_DEBUG("%s\n", busy_flag);
        return;
    }
    uint32_t res;
    uint8_t tout_s = argc >= 1 ? atoi(argv[0]) : 3;
    res = app_scan_start(tout_s);
    if (res == OB_ERROR_NO_ERR) {
        OM_LOG_DEBUG("Starting scan, timeout %d s\n", tout_s);
    } else {
        OM_LOG_DEBUG("Starting scan failed, error code = 0x%04X\n", res);
    }
}

static void cmd_shell_conn(int argc, char *argv[])
{
    uint8_t address[7];
    uint32_t res;
    if (argc <= 0) {
        OM_LOG_DEBUG("Usage(conn): conn [index in scan list] [timeout=5]\n");
        return;
    }
    if (busy_flag) {
        OM_LOG_DEBUG("%s\n", busy_flag);
        return;
    }
    int index = atoi(argv[0]);
    const char *dev_name;
    bool r = app_scan_device_get(index, address, &dev_name);
    if (!r) {
        OM_LOG_DEBUG("Device index %d invalid, scan device first\n", index);
        return;
    }
    uint8_t tout_s = argc >= 2 ? atoi(argv[1]) : 5;
    res = app_conn_start(address, tout_s, dev_name);
    if (res == OB_ERROR_NO_ERR) {
        const uint8_t *a = address;
        OM_LOG_DEBUG("Starting connect to (%d) %02X:%02X:%02X:%02X:%02X:%02X  %s, timeout %d s\n",
                     a[0], a[6], a[5], a[4], a[3], a[2], a[1], dev_name ? dev_name : "", tout_s);
    } else {
        OM_LOG_DEBUG("Starting connect failed, error code = 0x%04X\n", res);
    }
}

static void cmd_shell_disc(int argc, char *argv[])
{
    uint32_t res;
    struct app_device dev_buffer[OB_LE_HOST_CONNECTION_NB];
    uint8_t max_num = OB_LE_HOST_CONNECTION_NB;
    app_conn_device_get(dev_buffer, &max_num);
    if (argc <= 0) {
        OM_LOG_DEBUG("Usage(disc): disc [index in conn list]\n");
        if (max_num == 0) {
            OM_LOG_DEBUG("\tNo connected devices\n");
            return;
        }
        for (int i = 0; i < max_num; i++) {
            uint8_t *a = &dev_buffer[i].addr_type;
            OM_LOG_DEBUG("\tIdx: %d, (%d) %02X:%02X:%02X:%02X:%02X:%02X  %s\n",
                         dev_buffer[i].conn_idx, a[0], a[6], a[5], a[4], a[3], a[2], a[1], dev_buffer[i].name);
        }
        return;
    }
    if (busy_flag) {
        OM_LOG_DEBUG("%s\n", busy_flag);
        return;
    }
    int index = atoi(argv[0]);

    for (int i = 0; i < max_num; i++) {
        if (dev_buffer[i].conn_idx == index) {
            res = ob_gap_disconnect(index, 0x13);
            if (res == OB_ERROR_NO_ERR) {
                uint8_t *a = &dev_buffer[i].addr_type;
                OM_LOG_DEBUG("Disconnecting to idx=%d (%d) %02X:%02X:%02X:%02X:%02X:%02X  %s\n",
                             index, a[0], a[6], a[5], a[4], a[3], a[2], a[1], dev_buffer[i].name);
            } else {
                OM_LOG_DEBUG("Disconnect failed, error code = 0x%04X\n", res);
            }
        }
        return;
    }
    OM_LOG_DEBUG("Disconnect failed, no device found(idx=%d)\n", index);
}

static void cmd_shell_gattda(int argc, char *argv[])
{
    if (argc <= 0) {
        OM_LOG_DEBUG("Usage(gattda): gattda con_idx\n");
        return;
    }
    int index = atoi(argv[0]);
    app_gatt_client_discover_all(index);
}

static void cmd_shell_gattds(int argc, char *argv[])
{
    if (argc <= 0) {
        OM_LOG_DEBUG("Usage(gattds): gattds con_idx [uuid]\n");
        OM_LOG_DEBUG("    discover all service:     gattds 0 start_hdl end_hdl\n");
        OM_LOG_DEBUG("    discover service by uuid: gattds 0 0x180A\n");
        OM_LOG_DEBUG("    Note: only 16bit uuid support in shell\n");
        return;
    }
    int index = atoi(argv[0]);
    uint16_t uuid = 0;
    if (argc == 2 && !strncmp(argv[1], "0x", 2)) {
        uuid = strtol(&argv[1][2], NULL, 16);
        app_gatt_discover_service_uuid(index, (uint8_t *)&uuid, 2);
    }
    if (argc == 3) {
        uint16_t start_hdl = 0xFFFF, end_hdl = 0;
        if (!strncmp(argv[1], "0x", 2)) {
            start_hdl = strtol(&argv[1][2], NULL, 16);
        } else {
            start_hdl = strtol(argv[1], NULL, 10);
        }
        if (!strncmp(argv[2], "0x", 2)) {
            end_hdl = strtol(&argv[2][2], NULL, 16);
        } else {
            end_hdl = strtol(argv[2], NULL, 10);
        }
        app_gatt_discover_service(index, start_hdl, end_hdl);
    }
}

static void cmd_shell_gattdc(int argc, char *argv[])
{
    if (argc != 3) {
        OM_LOG_DEBUG("Usage(gattdc): gattdc con_idx start_hdl end_hdl\n");
        OM_LOG_DEBUG("      example: gattdc 0 0x0001 0xffff\n");
        return;
    }
    int index = atoi(argv[0]);
    if (argc == 3) {
        uint16_t start_hdl = 0xFFFF, end_hdl = 0;
        if (!strncmp(argv[1], "0x", 2)) {
            start_hdl = strtol(&argv[1][2], NULL, 16);
        } else {
            start_hdl = strtol(argv[1], NULL, 10);
        }
        if (!strncmp(argv[2], "0x", 2)) {
            end_hdl = strtol(&argv[2][2], NULL, 16);
        } else {
            end_hdl = strtol(argv[2], NULL, 10);
        }
        app_gatt_discover_char(index, start_hdl, end_hdl);
    }
}

static void cmd_shell_gattdd(int argc, char *argv[])
{
    if (argc != 3) {
        OM_LOG_DEBUG("Usage(gattdd): gattdd con_idx start_hdl end_hdl\n");
        OM_LOG_DEBUG("      example: gattdd 0 0x0001 0xffff\n");
        return;
    }
    int index = atoi(argv[0]);
    if (argc == 3) {
        uint16_t start_hdl = 0xFFFF, end_hdl = 0;
        if (!strncmp(argv[1], "0x", 2)) {
            start_hdl = strtol(&argv[1][2], NULL, 16);
        } else {
            start_hdl = strtol(argv[1], NULL, 10);
        }
        if (!strncmp(argv[2], "0x", 2)) {
            end_hdl = strtol(&argv[2][2], NULL, 16);
        } else {
            end_hdl = strtol(argv[2], NULL, 10);
        }
        app_gatt_discover_desc(index, start_hdl, end_hdl);
    }
}

static void cmd_shell_gattrh(int argc, char *argv[])
{
    if (argc != 2) {
        OM_LOG_DEBUG("Usage(gattrh): gattrh con_idx att_handle\n");
        OM_LOG_DEBUG("      example: gattrh 0 3\n");
        return;
    }
    int index = atoi(argv[0]);
    if (argc == 2) {
        uint16_t att_handle;
        if (!strncmp(argv[1], "0x", 2)) {
            att_handle = strtol(&argv[1][2], NULL, 16);
        } else {
            att_handle = strtol(argv[1], NULL, 10);
        }
        app_gatt_read(index, att_handle);
    }
}

static void cmd_shell_gattru(int argc, char *argv[])
{
    if (argc != 2) {
        OM_LOG_DEBUG("Usage(gattru): gattru con_idx uuid\n");
        OM_LOG_DEBUG("      example: gattru 0 0x2a00\n");
        OM_LOG_DEBUG("    Note: only 16bit uuid support in shell\n");
        return;
    }
    int index = atoi(argv[0]);
    if (argc == 2) {
        uint16_t uuid;
        if (!strncmp(argv[1], "0x", 2)) {
            uuid = strtol(&argv[1][2], NULL, 16);
        } else {
            uuid = strtol(argv[1], NULL, 10);
        }
        app_gatt_read_uuid(index, (uint8_t *)&uuid, 2);
    }
}

static void cmd_shell_gattw(int type, int argc, char *argv[])
{
    if (argc != 3) {
        const char *cmd = type == OB_GATTC_WRITE_REQ ? "gattwr" : "gattwc";
        OM_LOG_DEBUG("Usage(%s): %s con_idx att_handle data\n", cmd, cmd);
        OM_LOG_DEBUG("      example: %s 0 3 0123456789abcdef\n", cmd);
        return;
    }
    int index = atoi(argv[0]);
    uint16_t att_handle;
    if (!strncmp(argv[1], "0x", 2)) {
        att_handle = strtol(&argv[1][2], NULL, 16);
    } else {
        att_handle = strtol(argv[1], NULL, 10);
    }
    uint8_t data[32] = {0}, *p = data;
    for (int i = 0; i < (int)strlen(argv[2]); i++) {
        int n = 0;
        if ('a' <= argv[2][i] && argv[2][i] <= 'f') {
            n = argv[2][i] - 'a' + 10;
        } else if ('A' <= argv[2][i] && argv[2][i] <= 'F') {
            n = argv[2][i] - 'a' + 10;
        } else {
            n = argv[2][i] - '0';
        }
        *p |= n;
        if (i & 1) {
            p++;
        } else {
            *p = *p << 4;
        }
    }
    ob_gattc_write(index, att_handle, type, data, p - data);
}

static void cmd_shell_gattwr(int argc, char *argv[])
{
    cmd_shell_gattw(OB_GATTC_WRITE_REQ, argc, argv);
}

static void cmd_shell_gattwc(int argc, char *argv[])
{
    cmd_shell_gattw(OB_GATTC_WRITE_CMD, argc, argv);
}

const static shell_cmd_t ble_multilink_shell_cmd[] = {
    { "adv",     cmd_shell_adv,     "Start/Stop advertise        Usage: adv start/stop [index=0]"   },
    { "scan",    cmd_shell_scan,    "Scan device                 Usage: scan [timeout=3]" },
    { "conn",    cmd_shell_conn,    "Connect device              Usage: conn [index]"  },
    { "disc",    cmd_shell_disc,    "Disconnect device           Usage: disc con_idx"  },
    { "gattda",  cmd_shell_gattda,  "GATT Client Discover All    Usage: gattc con_idx"  },
    { "gattds",  cmd_shell_gattds,  "GATT Discover Service       Usage: gattds con_idx start_hdl end_hdl"  },
    { "gattdc",  cmd_shell_gattdc,  "GATT Discover Character     Usage: gattdc con_idx start_hdl end_hdl"  },
    { "gattdd",  cmd_shell_gattdd,  "GATT Discover Descriptor    Usage: gattdd con_idx start_hdl end_hdl"  },
    { "gattrh",  cmd_shell_gattrh,  "GATT Read by handle         Usage: gattrh con_idx att_handle"  },
    { "gattru",  cmd_shell_gattru,  "GATT Read by UUID           Usage: gattru con_idx uuid"  },
    { "gattwr",  cmd_shell_gattwr,  "GATT Write Request          Usage: gattwr con_idx att_handle data"  },
    { "gattwc",  cmd_shell_gattwc,  "GATT Write Command          Usage: gattwc con_idx att_handle data"  },
    { NULL,      NULL,              NULL},     /* donot deleted */
};


/*********************************************************************
 * EXTERN FUNCTIONS
 */
void set_shell_busy(const char *busy_desc)
{
    busy_flag = busy_desc;
}

/*********************************************************************
 * LOCAL FUNCTIONS
 */

/*******************************************************************************
 * PUBLIC FUNCTIONS
 */
/**
 *******************************************************************************
 * @brief  Init shell module
 *******************************************************************************
 */
void app_shell_init(void)
{
    shell_init(ble_multilink_shell_cmd);
}


/** @} */
