/**
 * @file  examples/ble_app_simple_server/src/main.c
 * @brief  simple server
 * @date Wed, Sep  5, 2018  5:19:05 PM
 * @author Boling SW Team
 *
 * @addtogroup APP_SIMPLE_SERVER_MAIN main.c
 * @ingroup APP_SIMPLE_SERVER
 * @details simple server
 *
 * @{
 */

/*********************************************************************
 * INCLUDES
 */
#include "om.h"
#include "om_driver.h"
#include "shell.h"
#include "evt.h"
#include "pm.h"
#include "board.h"
#include "mbr.h"
#include "nvds.h"

#include "ob_config.h"
#include "omble.h"

/* Kernel includes. */
#include "cmsis_os2.h"

/*********************************************************************
 * MACROS
 */

#define EVENT_BLUETOOTH_MASK        0x0001
#define EVENT_SYSTEM_RESERVE_MASK   0x00FF

/*********************************************************************
 * TYPEDEFS
 */


/*********************************************************************
 * CONSTANTS
 */


/*********************************************************************
 * LOCAL VARIABLES
 */
static osEventFlagsId_t xEvtEvent = NULL;
static evt_timer_t evt_timer_0;

/*********************************************************************
 * GLOBAL VARIABLES
 */


/*********************************************************************
 * LOCAL FUNCTIONS
 */
void app_adv_init();
void app_sec_init();
void service_common_init(void);
void app_blmicro_dfu_init(void);
void ancs_client_init(void);
void app_media_hid_init(void);
void app_tspp_init(void);

/**
 *******************************************************************************
 * @brief  evt timer 0 handler
 *
 * @param[in] timer  timer
 * @param[in] param  param
 *******************************************************************************
 */
static void evt_timer_0_handler(evt_timer_t *timer, void *param)
{
    OM_LOG_DEBUG("evt timer: %08X\n", timer->time);
}

/**
 * @brief  bluetooth event handler
 **/
static void vEvtEventHandler(void)
{
    if(xEvtEvent)
        osEventFlagsSet(xEvtEvent, EVENT_BLUETOOTH_MASK);
}

/**
 * @brief  bluetooth schedule task
 *
 * @param[in] pvParameters  pv parameters
 **/
static void vEvtScheduleTask(void *argument)
{
    uint32_t uxBits;
    struct ob_stack_param param = {
        .max_connection = OB_LE_HOST_CONNECTION_NB,
        .max_ext_adv_set = OB_LE_HOST_ADV_SET_NUM,
        .max_att_mtu = OB_LE_HOST_ATT_MTU,
        .max_gatt_serv_num = OB_LE_HOST_MAX_GATT_SERV_NUM,
        .max_gatt_write_cache = OB_LE_HOST_ATT_WRITE_CACHE_SIZE,
        .smp_sc_support = OB_LE_HOST_SC_PAIRING,
    };

    nvds_init(0);
    omble_init(&param);

    app_adv_init();
    app_sec_init();
    service_common_init();
    app_blmicro_dfu_init();
    ancs_client_init();
    app_media_hid_init();
    app_tspp_init();

    uint32_t address, length;
    uint16_t crc16;
    mbr_read_part(PART_TYPE_APP, &address, &length, &crc16);
    OM_LOG_DEBUG("running(%d): 0x%08X, len: %d, crc: 0x%04X\n", drv_pmu_reboot_reason(), address, length, crc16);

    // simple event timer
    evt_timer_set(&evt_timer_0, 2000, EVT_TIMER_REPEAT, evt_timer_0_handler, NULL);

    // enable sleep
    pm_sleep_enable(true);
    pm_sleep_allow(PM_ID_SHELL);

    // Create event
    xEvtEvent = osEventFlagsNew(NULL);

    // set ke event callback
    evt_schedule_trigger_callback_set(vEvtEventHandler);

    while (1)
    {
        // schedule
        evt_schedule();

        // Wait for event
        uxBits = osEventFlagsWait(xEvtEvent, 0xFFFF, osFlagsWaitAny, osWaitForever);
        (void)uxBits;
    }
}

/*********************************************************************
 * PUBLIC FUNCTIONS
 */

/**
 * @brief  v start bluetooth task
 **/
void vStartEvtTask(void)
{
    const osThreadAttr_t bluetoothThreadAttr =
    {
        .name = NULL,
        .attr_bits = 0,
        .cb_mem = NULL,
        .cb_size = 0,
        .stack_mem = NULL,
        .stack_size = 2048,
        .priority = osPriorityRealtime,
        .tz_module = 0,
    };

    // Create ble Task
    osThreadNew(vEvtScheduleTask, NULL, &bluetoothThreadAttr);
}

/** @} */

// vim: fdm=marker
