#ifndef __T1001_SLEEP_H__
#define __T1001_SLEEP_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "t1001.h"

/**
 * @addtogroup T1001_StdPeriph_Driver
 * @{
 */

/**
 * @addtogroup SLEEP_Driver
 * @brief 	   SLEEP driver modules
 * @{
 */

/** @brief Enumerations for the possible microcontroller sleep modes.
 * - SLEEPMODE_RUNNING
 *     Everything is active and running.  In practice this mode is not
 *     used, but it is defined for completeness of information.
 * - SLEEPMODE_CPU_HALT
 *     Only the CPU is halted.  The rest of the chip continues running
 *     normally.  The chip will wake from any interrupt.
 * - SLEEPMODE_WAKETIMER
 *     The device is in deep sleep and the timer clock sources remain running.
 *     Wakeup is possible from both GPIO and the Hal Virtual Timers.
 * - SLEEPMODE_NOTIMER
 *     The device is in deep sleep. All the peripherals and clock sources are turned off.
 *     Wakeup is possible only from GPIOs IO9-IO10-IO11-IO12-IO13.
 */
typedef enum {
    SLEEPMODE_RUNNING       = 0,
    SLEEPMODE_CPU_HALT      = 1,
    SLEEPMODE_WAKETIMER     = 2,
    SLEEPMODE_NOTIMER       = 3,
} SleepModes;

/**
 * @brief This function allows to put the T1001 in low power state.
 *
 * This function allows to enter a desired sleep mode that is negotiated between BT stack needs and application needs.
 * The application can provide the desired sleep mode using the parameter sleepMode. In addition to this, the application can
 * optionally use the @ref SLEEP_AppSleepModeCheck to provide its desired sleep mode.
 * The function will compute the sleep mode by combining the different requests and choosing the lowest possible power mode.
 *
 * The device can be configured with different wake sources
 * according to the sleepMode parameter. The lowest power mode is obtained configuring the device in
 * SLEEPMODE_NOTIMER.
 * For all sleep modes, T1001_Sleep() will return when the wakeup occours.
 *
 * @param sleepMode Sleep mode (see SleepModes enum)
 *
 * @param gpioWakeBitMask  A bit mask of the GPIO that are allowed to wake
 * the chip from deep sleep.  A high bit in the mask will enable waking
 * the chip if the corresponding GPIO changes state.
 * @param gpioWakeLevelMask  A mask used to setup the active wakeup level:
 * - 0: the system wakes up when IO is low
 * - 1: the system wakes up when IO is high
 * The level sensitive bit mask is the same of the gpioWakeBitMask parameter.
 *
 * @retval Status of the call
 */
uint8_t T1001_Sleep(SleepModes sleepMode, uint32_t gpioWakeBitMask, uint32_t gpionWakeBitMask);

/**
 * @brief This function allows the application to define its desired sleep mode.
 *
 * The App_SleepMode_Check allows the application to set its desired sleep mode based on the application power management policy.
 * When user calls @ref T1001_Sleep, a negotiation occurs to define the sleep mode and this function is called to get
 * inputs from application.
 * It is important to notice that this function is executed with interrupts disabled
 *
 * @param sleepMode Sleep mode (see SleepModes enum)
 *
 * @retval Return the sleep mode possible and safe from the application point of view
 *
 * @note A weak implementation always returning SLEEPMODE_NOTIMER is provided as default
 * when no application specifc behaviour is required.
 */

SleepModes App_SleepMode_Check(SleepModes sleepMode);
uint32_t t1001_sleep_get_protect_time();
void omw_sleep_set_force_sleep();
uint32_t omw_sleep_clr_wkup_rec();

#ifdef __cplusplus
}
#endif
#endif