/*
 * Copyright (C) 2014 BlueKitchen GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 * 4. Any redistribution, use, or modification is done solely for
 *    personal benefit and not for any commercial purpose or for
 *    monetary gain.
 *
 * THIS SOFTWARE IS PROVIDED BY BLUEKITCHEN GMBH AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BLUEKITCHEN
 * GMBH OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
 * THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Please inquire about commercial licensing options at 
 * contact@bluekitchen-gmbh.com
 *
 */


/**
 * @title ATT Dispatch
 *
 * Dispatcher for independent implementation of ATT client and server.
 *
 */

#ifndef ATT_DISPATCH_H
#define ATT_DISPATCH_H

#include "btstack_util.h"

#if defined __cplusplus
extern "C" {
#endif

/**
 * @brief Register packet handler for ATT Client PDUs
 * @param packet_handler
 */
void att_dispatch_register_client(btstack_packet_handler_t packet_handler);

/**
 * @brief Register packet handler for ATT Server PDUs
 * @param packet_handler
 */
void att_dispatch_register_server(btstack_packet_handler_t packet_handler);

/**
 * @brief Register L2CAP ATT Service with fixed PSM
 * @note requires ENABLE_GATT_OVER_CLASSIC
 */
void att_dispatch_classic_register_service(void);

/**
 * @brief Create L2CAP ATT Connection
 * @note ENABLE_GATT_OVER_CLASSIC
 * @param address
 * @param l2cap_psm
 */
uint8_t att_dispatch_classic_connect(bd_addr_t address, uint16_t l2cap_psm, uint16_t *out_cid);

/**
 * @brief can send packet for client
 * @param con_handle
 */
bool att_dispatch_client_can_send_now(hci_con_handle_t con_handle);

/**
 * @brief can send packet for server
 * @param con_handle
 */
bool att_dispatch_server_can_send_now(hci_con_handle_t con_handle);

/** 
 * @brief Request emission of L2CAP_EVENT_CAN_SEND_NOW as soon as possible for client
 * @note L2CAP_EVENT_CAN_SEND_NOW might be emitted during call to this function
 *       so packet handler should be ready to handle it
 * @param con_handle
 */
void att_dispatch_client_request_can_send_now_event(hci_con_handle_t con_handle);

/** 
 * @brief Request emission of L2CAP_EVENT_CAN_SEND_NOW as soon as possible for server
 * @note L2CAP_EVENT_CAN_SEND_NOW might be emitted during call to this function
 *       so packet handler should be ready to handle it
 * @param con_handle
 */
void att_dispatch_server_request_can_send_now_event(hci_con_handle_t con_handle);

/** 
* @brief Used for propogating a updated MTU from att_server to gatt_client
* @param con_handle
* @param mtu
*/
void att_dispatch_server_mtu_exchanged(hci_con_handle_t con_handle, uint16_t new_mtu);

/**
* @brief Used for propogating a updated MTU from gatt_client to att_server
* @param con_handle
* @param mtu
*/
void att_dispatch_client_mtu_exchanged(hci_con_handle_t con_handle, uint16_t new_mtu);

#if defined __cplusplus
}
#endif

#endif // ATT_DISPATCH_H
