#ifndef T100X_HAL_2G4_H
#define T100X_HAL_2G4_H
#include "t100x_2g4_def.h"
#include "t100x_hal_rf.h"

/** \brief Initialize 2.4G parameters
 *  \param param */
void omw_svc_2g4_init(omw_2g4_param_t* param);

/** \brief 2.4G send buffer data
 *
 * #ifndef USE_TBL_TO_SET_FREQ
 *
 * support freq:   2364M ~ 2520M,
 * channel width:  100K, 500K, 1M, 2M
 *
 * channel idx = (freq - 2364M) / width
 * channel idx[100K]:  0 ~ 1559
 * channel idx[500K]:  0 ~ 311
 * channel idx[1M]:    0 ~ 155
 * channel idx[2M]:    0 ~ 77
 *
 * rf_chl: bits[15:11] channel width,  0=default 1M,  1=100K, 5=500K, 10=1M, 20=2M
 * rf_chl: bits[10:0]  channel idx
 * whiten_chl: for whiten, such as 37/38/39
 *
 * #else
 *
 * [like BLE]
 * support freq: 2402 ~ 2480
 * channel idx:  0 ~ 39 [1M, 2M same]
 * rf_chl: rf channel idx, whiten channel will be map to BLE Channel automaticlly, such as 0=37, 12=38, 39=39
 *
 * #endif
 *
 * tx/rx can not cfg at the same time
 *  \param data Buffer data
 *  \param len Buffer length in bytes
 *  \param rf_chl RF Channel(0 ~ 39)
 *  \param whiten_chl Whitening initial value(if NOT USE_TBL_TO_SET_FREQ) */
void omw_svc_2g4_tx_data(uint8_t * data, uint16_t len, uint16_t rf_chl
                        #ifndef USE_TBL_TO_SET_FREQ
                        , uint8_t whiten_chl
                        #endif
                        );

/** \brief 2.4G receive data to buffer
 *  \param data Buffer
 *  \param len RX length in bytes
 *  \param rf_chl RF Channel(0 ~ 39)
 *  \param whiten_chl Whitening initial value(if NOT USE_TBL_TO_SET_FREQ) */
void omw_svc_2g4_rx_data(uint8_t * data, uint16_t len, uint16_t rf_chl
                        #ifndef USE_TBL_TO_SET_FREQ
                        , uint8_t whiten_chl
                        #endif
                        );

/** Set RF to IDLE state, current operation will be aborted */
void omw_svc_2g4_set_idle(void);

/** \brief Check if the RF module is idle.
 *  \return
 * 0: NOT idle; !0: Idle
 *
 * If call this function, must after you HAVE START TX/RX */
uint8_t omw_svc_24g_is_txrx_idle();

/** \brief Entering sleep state. App will be blocked until wake up
 *  \return
 * 0: Woke up by timeout.
 *
 * !0: Woke up by GPIO trigger. */
uint32_t omw_sleep_goto_sleep();

/** \brief Set sleeping timeout
 *  \param slp_ms Timeout value in microseconds */
void omw_sleep_set_sleep_len(uint32_t slp_ms);

/** \brief Toggle whitening state of TX
 *  \param en 1: Enable; 0: Disable */
void omw_svc_2g4_en_whiten_tx(uint8_t en);

/** \brief Toggle whitening state of RX
 *  \param en 1: Enable; 0: Disable */
void omw_svc_2g4_en_whiten_rx(uint8_t en);

/** \brief Retrieve 2.4G FIFO buffer
 *  \return Pointer of the buffer */
uint8_t *omw_svc_2g4_get_fifo(void);

/** \brief Retrieve 2.4G parameter
 *  \return Pointer of the parameter */
omw_2g4_param_t *omw_svc_2g4_get_param(void);

/** \brief Update rx wait sync timeout val
 *  \param to_val timeout val, us */
void omw_svc_2g4_update_rx_to(uint32_t to_val);

/** \brief Update access word
 *  \param acc_word access word value */
void omw_svc_2g4_set_access_word(uint32_t acc_word);

#endif // T100X_HAL_2G4_H