#ifndef __BT_LOG_H__
#define __BT_LOG_H__

#include "omw_config.h"

#include "base/bt_common.h"

#define LOG_IMPL_LEVEL_OFF  0U
#define LOG_IMPL_LEVEL_ERR  1U
#define LOG_IMPL_LEVEL_WRN  2U
#define LOG_IMPL_LEVEL_INF  3U
#define LOG_IMPL_LEVEL_DBG  4U

#ifdef CONFIG_BT_DEBUG
#define LOG_LEVEL CONFIG_BT_LOG_LEVEL
#else
#define LOG_LEVEL LOG_IMPL_LEVEL_OFF
#endif

static inline char z_log_minimal_level_to_char(int level)
{
    switch (level)
    {
    case LOG_IMPL_LEVEL_ERR:
        return 'E';
    case LOG_IMPL_LEVEL_WRN:
        return 'W';
    case LOG_IMPL_LEVEL_INF:
        return 'I';
    case LOG_IMPL_LEVEL_DBG:
        return 'D';
    default:
        return '?';
    }
}

extern void bt_log_impl_printf(uint8_t _level, const char *format, ...);
extern void bt_log_impl_packet(uint8_t packet_type, uint8_t in, uint8_t *packet, uint16_t len);
extern void bt_log_impl_point(uint32_t val);
extern void bt_log_impl_init(void);

#define LOG_IMPL_TO_PRINTK(_fun, _line, _level, _name, fmt, ...)                                   \
    do                                                                                             \
    {                                                                                              \
        bt_log_impl_printf(_level,                                                                 \
                           "%c: "                                                                  \
                           "(%s)"                                                                  \
                           "%s():%d: " fmt "\n",                                                   \
                           z_log_minimal_level_to_char(_level), #_name, _fun, _line,               \
                           ##__VA_ARGS__);                                                         \
    } while (false);

#ifdef CONFIG_BT_DEBUG
#define __LOG_IMPL(_level, _name, _level_thod, ...)                                                \
    if (_level <= _level_thod)                                                                     \
    {                                                                                              \
        LOG_IMPL_TO_PRINTK(__func__, __LINE__, _level, _name, __VA_ARGS__);                        \
    }

#define __LOG_IMPL_RAW(_level, _fmt, ...)                                                          \
    bt_log_impl_printf(_level, _fmt, ##__VA_ARGS__);

#ifdef CONFIG_BT_DEBUG_PACKET
#define __PACKET_IMPL(_packet_type, _in, _packet, _len)                                            \
    bt_log_impl_packet(_packet_type, _in, _packet, _len);
#define __LOG_POINT_IMPL(_val)                                                                     \
    bt_log_impl_point(_val);
#else
#define __PACKET_IMPL(_packet_type, _in, _packet, _len)
#define __LOG_POINT_IMPL(_val)
#endif
#define __LOG_INIT_IMPL()                                                                          \
    bt_log_impl_init(_packet_type, _in, _packet, _len);
#else
#define __LOG_IMPL(_level, _name, _level_thod, ...)
#define __LOG_IMPL_RAW(_level, _fmt, ...)
#define __PACKET_IMPL(_packet_type, _in, _packet, _len)
#define __LOG_POINT_IMPL(_val)
#define __LOG_INIT_IMPL()
#endif

static inline char *log_strdup(const char *str)
{
    return (char *)str;
}

/**
 * @brief Logger API
 * @defgroup log_api Logging API
 * @ingroup logger
 * @{
 */

/**
 * @brief Writes an ERROR level message to the log.
 *
 * @details It's meant to report severe errors, such as those from which it's
 * not possible to recover.
 *
 * @param ... A string optionally containing printk valid conversion specifier,
 * followed by as many values as specifiers.
 */
#define LOG_IMPL_ERR(...) __LOG_IMPL(LOG_IMPL_LEVEL_ERR, __VA_ARGS__)

/**
 * @brief Writes a WARNING level message to the log.
 *
 * @details It's meant to register messages related to unusual situations that
 * are not necessarily errors.
 *
 * @param ... A string optionally containing printk valid conversion specifier,
 * followed by as many values as specifiers.
 */
#define LOG_IMPL_WRN(...) __LOG_IMPL(LOG_IMPL_LEVEL_WRN, __VA_ARGS__)

/**
 * @brief Writes an INFO level message to the log.
 *
 * @details It's meant to write generic user oriented messages.
 *
 * @param ... A string optionally containing printk valid conversion specifier,
 * followed by as many values as specifiers.
 */
#define LOG_IMPL_INF(...) __LOG_IMPL(LOG_IMPL_LEVEL_INF, __VA_ARGS__)

/**
 * @brief Writes a DEBUG level message to the log.
 *
 * @details It's meant to write developer oriented information.
 *
 * @param ... A string optionally containing printk valid conversion specifier,
 * followed by as many values as specifiers.
 */
#define LOG_IMPL_DBG(...) __LOG_IMPL(LOG_IMPL_LEVEL_DBG, __VA_ARGS__)

#define LOG_INIT() __LOG_INIT_IMPL()

#define LOG_PACKET_DUMP(_packet_type, _in, _packet, _len)                                          \
    __PACKET_IMPL(_packet_type, _in, _packet, _len)



#define LOG_DBG(fmt, ...)  LOG_IMPL_DBG(LOG_MODULE_NAME, LOG_LEVEL, fmt, ##__VA_ARGS__)
#define LOG_ERR(fmt, ...)  LOG_IMPL_ERR(LOG_MODULE_NAME, LOG_LEVEL, fmt, ##__VA_ARGS__)
#define LOG_WRN(fmt, ...) LOG_IMPL_WRN(LOG_MODULE_NAME, LOG_LEVEL, fmt, ##__VA_ARGS__)
#define LOG_INF(fmt, ...) LOG_IMPL_INF(LOG_MODULE_NAME, LOG_LEVEL, fmt, ##__VA_ARGS__)

#define BT_PACKET_DUMP(_packet_type, _in, _packet, _len)                                           \
    LOG_PACKET_DUMP(_packet_type, _in, _packet, _len)


#define Z_LOG_HEXDUMP(_level, _data, _length, _str) \
    LOG_IMPL_DBG(LOG_MODULE_NAME, LOG_LEVEL, _str "%s", bt_hex_real(_data, _length))

/**
 * @brief Writes an ERROR level hexdump message to the log.
 *
 * @details It's meant to report severe errors, such as those from which it's
 * not possible to recover.
 *
 * @param _data   Pointer to the data to be logged.
 * @param _length Length of data (in bytes).
 * @param _str    Persistent, raw string.
 */
#define LOG_HEXDUMP_ERR(_data, _length, _str) \
    LOG_IMPL_ERR(LOG_MODULE_NAME, LOG_LEVEL, _str "%s", bt_hex_real(_data, _length))
	// Z_LOG_HEXDUMP(LOG_LEVEL_ERR, _data, _length, _str)

/**
 * @brief Writes a WARNING level message to the log.
 *
 * @details It's meant to register messages related to unusual situations that
 * are not necessarily errors.
 *
 * @param _data   Pointer to the data to be logged.
 * @param _length Length of data (in bytes).
 * @param _str    Persistent, raw string.
 */
#define LOG_HEXDUMP_WRN(_data, _length, _str) \
    LOG_IMPL_WRN(LOG_MODULE_NAME, LOG_LEVEL, _str "%s", bt_hex_real(_data, _length))
	// Z_LOG_HEXDUMP(LOG_LEVEL_WRN, _data, _length, _str)

/**
 * @brief Writes an INFO level message to the log.
 *
 * @details It's meant to write generic user oriented messages.
 *
 * @param _data   Pointer to the data to be logged.
 * @param _length Length of data (in bytes).
 * @param _str    Persistent, raw string.
 */
#define LOG_HEXDUMP_INF(_data, _length, _str) \
    LOG_IMPL_INF(LOG_MODULE_NAME, LOG_LEVEL, _str "%s", bt_hex_real(_data, _length))
	// Z_LOG_HEXDUMP(LOG_LEVEL_INF, _data, _length, _str)

/**
 * @brief Writes a DEBUG level message to the log.
 *
 * @details It's meant to write developer oriented information.
 *
 * @param _data   Pointer to the data to be logged.
 * @param _length Length of data (in bytes).
 * @param _str    Persistent, raw string.
 */
#define LOG_HEXDUMP_DBG(_data, _length, _str) \
    LOG_IMPL_DBG(LOG_MODULE_NAME, LOG_LEVEL, _str "%s", bt_hex_real(_data, _length))
	// Z_LOG_HEXDUMP(LOG_LEVEL_DBG, _data, _length, _str)



#define BT_HEXDUMP_DBG(_data, _length, _str) LOG_HEXDUMP_DBG((const uint8_t *)_data, _length, _str)


#define BT_LOG_POINT(LOG_LEVEL,TYPE_STR,STR,FLAG,VALUE)  __LOG_POINT_IMPL((((FLAG) << 16) | ((uint16_t)VALUE)))


#define BT_LOG_POINT_2(LOG_LEVEL,TYPE_STR,STR,FLAG,VALUE_0,VALUE_1)  \
	BT_LOG_POINT(LOG_LEVEL,TYPE_STR,STR,FLAG,(((uint8_t)(VALUE_0) & 0xff)<<8)|(((uint8_t)(VALUE_1) & 0xff)<<0))
#define BT_LOG_POINT_4(LOG_LEVEL,TYPE_STR,STR,FLAG,VALUE_0,VALUE_1,VALUE_2,VALUE_3) \
	BT_LOG_POINT(LOG_LEVEL,TYPE_STR,STR,FLAG,(((uint8_t)(VALUE_0) & 0x0f)<<12)|(((uint8_t)(VALUE_1) & 0x0f)<<8)|(((uint8_t)(VALUE_2) & 0x0f)<<4)|(((uint8_t)(VALUE_3) & 0x0f)<<0))
#define BT_LOG_POINT_8(LOG_LEVEL,TYPE_STR,STR,FLAG,VALUE_0,VALUE_1,VALUE_2,VALUE_3,VALUE_4,VALUE_5,VALUE_6,VALUE_7)\
	BT_LOG_POINT(LOG_LEVEL,TYPE_STR,STR,FLAG,(((uint8_t)(VALUE_0) & 0x03)<<14)|(((uint8_t)(VALUE_1) & 0x03)<<12)|(((uint8_t)(VALUE_2) & 0x03)<<10)|(((uint8_t)(VALUE_3) & 0x03)<<8)|(((uint8_t)(VALUE_4) & 0x03)<<6)|(((uint8_t)(VALUE_5) & 0x03)<<4)|(((uint8_t)(VALUE_6) & 0x03)<<2)|(((uint8_t)(VALUE_7) & 0x07)<<0))

#define POINT_LEVEL_NORMAL    (0)
#define POINT_LEVEL_INFO      (1)
#define POINT_LEVEL_CORE      (2)
#define POINT_LEVEL_ERROR     (3)


// Debug Log Point Define, we should use log function by those macro
// To judge use, in respin, we only allow use point 0x1000-0x8FFF
// To judge use, in cm0, we only allow use point 0x9000-0xEFFF
// 0xF000-0xFFFF reserve for some special use case
enum {
	// For Controller 0x1000-0x3fff
	// For Controller BB
	LOG_POINT_1000 = 0x1000,
	LOG_POINT_1001 = 0x1001,
	LOG_POINT_1002 = 0x1002,
	LOG_POINT_1003 = 0x1003,
	LOG_POINT_1004 = 0x1004,
	LOG_POINT_1005 = 0x1005,
	LOG_POINT_1006 = 0x1006,
	LOG_POINT_1007 = 0x1007,
	LOG_POINT_1008 = 0x1008,
	LOG_POINT_1009 = 0x1009,
	LOG_POINT_100A = 0x100A,
	LOG_POINT_100B = 0x100B,
	LOG_POINT_100C = 0x100C,
	LOG_POINT_100D = 0x100D,
	LOG_POINT_100E = 0x100E,
	LOG_POINT_100F = 0x100F,

	LOG_POINT_1010 = 0x1010,
	LOG_POINT_1011 = 0x1011,
	LOG_POINT_1012 = 0x1012,
	LOG_POINT_1013 = 0x1013,
	LOG_POINT_1014 = 0x1014,
	LOG_POINT_1015 = 0x1015,
	LOG_POINT_1016 = 0x1016,
	LOG_POINT_1017 = 0x1017,
	LOG_POINT_1018 = 0x1018,
	LOG_POINT_1019 = 0x1019,
	LOG_POINT_101A = 0x101A,
	LOG_POINT_101B = 0x101B,
	LOG_POINT_101C = 0x101C,
	LOG_POINT_101D = 0x101D,
	LOG_POINT_101E = 0x101E,
	LOG_POINT_101F = 0x101F,

    LOG_POINT_1100 = 0x1100,
    LOG_POINT_1101 = 0x1101,
    LOG_POINT_1102 = 0x1102,
    LOG_POINT_1103 = 0x1103,
    LOG_POINT_1104 = 0x1104,
    LOG_POINT_1105 = 0x1105,
    LOG_POINT_1106 = 0x1106,
    LOG_POINT_1107 = 0x1107,
    LOG_POINT_1108 = 0x1108,
    LOG_POINT_1109 = 0x1109,
    LOG_POINT_110A = 0x110A,
    LOG_POINT_110B = 0x110B,
    LOG_POINT_110C = 0x110C,
    LOG_POINT_110D = 0x110D,
    LOG_POINT_110E = 0x110E,
    LOG_POINT_110F = 0x110F,

    LOG_POINT_1110 = 0x1110,
    LOG_POINT_1111 = 0x1111,
    LOG_POINT_1112 = 0x1112,
    LOG_POINT_1113 = 0x1113,
    LOG_POINT_1114 = 0x1114,
    LOG_POINT_1115 = 0x1115,
    LOG_POINT_1116 = 0x1116,
    LOG_POINT_1117 = 0x1117,
    LOG_POINT_1118 = 0x1118,
    LOG_POINT_1119 = 0x1119,
    LOG_POINT_111A = 0x111A,
    LOG_POINT_111B = 0x111B,
    LOG_POINT_111C = 0x111C,
    LOG_POINT_111D = 0x111D,
    LOG_POINT_111E = 0x111E,
    LOG_POINT_111F = 0x111F,

    LOG_POINT_1120 = 0x1120,
    LOG_POINT_1121 = 0x1121,

    // SCAN
    LOG_POINT_1200 = 0x1200,
    LOG_POINT_1201 = 0x1201,
    LOG_POINT_1202 = 0x1202,
    LOG_POINT_1203 = 0x1203,
    LOG_POINT_1204 = 0x1204,
    LOG_POINT_1205 = 0x1205,
    LOG_POINT_1206 = 0x1206,
    LOG_POINT_1207 = 0x1207,
    LOG_POINT_1208 = 0x1208,
    LOG_POINT_1209 = 0x1209,
    LOG_POINT_120A = 0x120A,
    LOG_POINT_120B = 0x120B,
    LOG_POINT_120C = 0x120C,
    LOG_POINT_120D = 0x120D,
    LOG_POINT_120E = 0x120E,
    LOG_POINT_120F = 0x120F,

    // ADV
    LOG_POINT_1300 = 0x1300,
    LOG_POINT_1301 = 0x1301,
    LOG_POINT_1302 = 0x1302,
    LOG_POINT_1303 = 0x1303,
    LOG_POINT_1304 = 0x1304,
    LOG_POINT_1305 = 0x1305,
    LOG_POINT_1306 = 0x1306,
    LOG_POINT_1307 = 0x1307,
    LOG_POINT_1308 = 0x1308,
    LOG_POINT_1309 = 0x1309,
    LOG_POINT_130A = 0x130A,
    LOG_POINT_130B = 0x130B,
    LOG_POINT_130C = 0x130C,
    LOG_POINT_130D = 0x130D,
    LOG_POINT_130E = 0x130E,
    LOG_POINT_130F = 0x130F,


	// For Controller HCI
	LOG_POINT_2000 = 0x2000,
	LOG_POINT_2001 = 0x2001,
	LOG_POINT_2002 = 0x2002,
	LOG_POINT_2003 = 0x2003,
	LOG_POINT_2004 = 0x2004,
	LOG_POINT_2005 = 0x2005,
	LOG_POINT_2006 = 0x2006,
	LOG_POINT_2007 = 0x2007,
	LOG_POINT_2008 = 0x2008,
	LOG_POINT_2009 = 0x2009,
	LOG_POINT_200A = 0x200A,
	LOG_POINT_200B = 0x200B,
	LOG_POINT_200C = 0x200C,
	LOG_POINT_200D = 0x200D,
	LOG_POINT_200E = 0x200E,
	LOG_POINT_200F = 0x200F,

    LOG_POINT_2100 = 0x2100,
    LOG_POINT_2101 = 0x2101,






	// For IPC info
	LOG_POINT_9000 = 0x9000,
	LOG_POINT_9001 = 0x9001,
	LOG_POINT_9002 = 0x9002,
	LOG_POINT_9003 = 0x9003,
	LOG_POINT_9004 = 0x9004,
	LOG_POINT_9005 = 0x9005,
	LOG_POINT_9006 = 0x9006,
	LOG_POINT_9007 = 0x9007,
	LOG_POINT_9008 = 0x9008,
	LOG_POINT_9009 = 0x9009,
	LOG_POINT_900A = 0x900A,
	LOG_POINT_900B = 0x900B,
	LOG_POINT_900C = 0x900C,
	LOG_POINT_900D = 0x900D,
	LOG_POINT_900E = 0x900E,
	LOG_POINT_900F = 0x900F,

	LOG_POINT_9010 = 0x9010,
	LOG_POINT_9011 = 0x9011,
	LOG_POINT_9012 = 0x9012,
	LOG_POINT_9013 = 0x9013,
	LOG_POINT_9014 = 0x9014,
	LOG_POINT_9015 = 0x9015,
	LOG_POINT_9016 = 0x9016,
	LOG_POINT_9017 = 0x9017,
	LOG_POINT_9018 = 0x9018,
	LOG_POINT_9019 = 0x9019,
	LOG_POINT_901A = 0x901A,
	LOG_POINT_901B = 0x901B,
	LOG_POINT_901C = 0x901C,
	LOG_POINT_901D = 0x901D,
	LOG_POINT_901E = 0x901E,
	LOG_POINT_901F = 0x901F,

	LOG_POINT_9020 = 0x9020,
	LOG_POINT_9021 = 0x9021,
	LOG_POINT_9022 = 0x9022,


	LOG_POINT_9028 = 0x9028,
	LOG_POINT_9029 = 0x9029,

	LOG_POINT_9030 = 0x9030,
	LOG_POINT_9031 = 0x9031,
	LOG_POINT_9032 = 0x9032,
	LOG_POINT_9033 = 0x9033,

	// For UI info
	// Voice
	LOG_POINT_A000 = 0xA000,
	LOG_POINT_A001 = 0xA001,
	LOG_POINT_A002 = 0xA002,
	LOG_POINT_A003 = 0xA003,
	LOG_POINT_A004 = 0xA004,
	LOG_POINT_A005 = 0xA005,
	LOG_POINT_A006 = 0xA006,
	LOG_POINT_A007 = 0xA007,
	LOG_POINT_A008 = 0xA008,
	LOG_POINT_A009 = 0xA009,
	LOG_POINT_A00A = 0xA00A,
	LOG_POINT_A00B = 0xA00B,
	LOG_POINT_A00C = 0xA00C,
	LOG_POINT_A00D = 0xA00D,
	LOG_POINT_A00E = 0xA00E,
	LOG_POINT_A00F = 0xA00F,
	// Led
	LOG_POINT_A100 = 0xA100,
	LOG_POINT_A101 = 0xA101,
	LOG_POINT_A102 = 0xA102,
	// Battery
	LOG_POINT_A200 = 0xA200,
	LOG_POINT_A201 = 0xA201,
	LOG_POINT_A202 = 0xA202,
	LOG_POINT_A203 = 0xA203,
	LOG_POINT_A204 = 0xA204,
	LOG_POINT_A205 = 0xA205,
	LOG_POINT_A206 = 0xA206,
	LOG_POINT_A207 = 0xA207,
	LOG_POINT_A208 = 0xA208,
	LOG_POINT_A209 = 0xA209,
	LOG_POINT_A20A = 0xA20A,
	LOG_POINT_A20B = 0xA20B,

	LOG_POINT_A210 = 0xA210,
	LOG_POINT_A211 = 0xA211,
	LOG_POINT_A212 = 0xA212,
	LOG_POINT_A213 = 0xA213,
	LOG_POINT_A214 = 0xA214,
	LOG_POINT_A215 = 0xA215,
	LOG_POINT_A216 = 0xA216,
	LOG_POINT_A217 = 0xA217,
	LOG_POINT_A218 = 0xA218,
	LOG_POINT_A219 = 0xA219,
	LOG_POINT_A21A = 0xA21A,
	// Key
	LOG_POINT_A300 = 0xA300,
	LOG_POINT_A301 = 0xA301,
	LOG_POINT_A302 = 0xA302,
	// BLE ADV
	LOG_POINT_A400 = 0xA400,
	// Timer
	LOG_POINT_A500 = 0xA500,
	LOG_POINT_A501 = 0xA501,
	LOG_POINT_A502 = 0xA502,
	LOG_POINT_A503 = 0xA503,
	LOG_POINT_A504 = 0xA504,
	LOG_POINT_A505 = 0xA505,

	// Audio
	LOG_POINT_A600 = 0xA600,
	LOG_POINT_A601 = 0xA601,
	LOG_POINT_A602 = 0xA602,
	LOG_POINT_A603 = 0xA603,
	LOG_POINT_A604 = 0xA604,
	LOG_POINT_A605 = 0xA605,
	LOG_POINT_A606 = 0xA606,
	LOG_POINT_A607 = 0xA607,
	LOG_POINT_A608 = 0xA608,
	LOG_POINT_A609 = 0xA609,
	LOG_POINT_A60A = 0xA60A,
	LOG_POINT_A60B = 0xA60B,
	LOG_POINT_A60C = 0xA60C,
	LOG_POINT_A60D = 0xA60D,
	LOG_POINT_A60E = 0xA60E,
	LOG_POINT_A60F = 0xA60F,
	LOG_POINT_A610 = 0xA610,
	LOG_POINT_A611 = 0xA611,
	LOG_POINT_A612 = 0xA612,
	LOG_POINT_A613 = 0xA613,
	LOG_POINT_A620 = 0xA620,
	LOG_POINT_A621 = 0xA621,
	LOG_POINT_A622 = 0xA622,
	LOG_POINT_A623 = 0xA623,
	LOG_POINT_A624 = 0xA624,

	LOG_POINT_A630 = 0xA630,
	LOG_POINT_A631 = 0xA631,
	LOG_POINT_A632 = 0xA632,
	LOG_POINT_A633 = 0xA633,
	LOG_POINT_A634 = 0xA634,
	LOG_POINT_A635 = 0xA635,

// mSBC
	LOG_POINT_A800 = 0xA800,
	LOG_POINT_A801 = 0xA801,
	LOG_POINT_A802 = 0xA802,
	LOG_POINT_A803 = 0xA803,
	LOG_POINT_A804 = 0xA804,
	LOG_POINT_A805 = 0xA805,
	LOG_POINT_A806 = 0xA806,
	LOG_POINT_A807 = 0xA807,
	// TWS
	// Audio sync
	LOG_POINT_B000 = 0xB000,
	LOG_POINT_B001 = 0xB001,
	LOG_POINT_B002 = 0xB002,
	LOG_POINT_B003 = 0xB003,
	LOG_POINT_B004 = 0xB004,
	LOG_POINT_B005 = 0xB005,
	LOG_POINT_B006 = 0xB006,
	LOG_POINT_B007 = 0xB007,
	LOG_POINT_B008 = 0xB008,
	LOG_POINT_B009 = 0xB009,
	LOG_POINT_B00A = 0xB00A,
	LOG_POINT_B00B = 0xB00B,
	LOG_POINT_B00C = 0xB00C,
	LOG_POINT_B00D = 0xB00D,
	LOG_POINT_B00E = 0xB00E,
	LOG_POINT_B00F = 0xB00F,
	LOG_POINT_B010 = 0xB010,
	LOG_POINT_B011 = 0xB011,
	LOG_POINT_B012 = 0xB012,
	LOG_POINT_B013 = 0xB013,
	LOG_POINT_B014 = 0xB014,
	LOG_POINT_B015 = 0xB015,
	LOG_POINT_B016 = 0xB016,
	LOG_POINT_B017 = 0xB017,
	LOG_POINT_B018 = 0xB018,

	LOG_POINT_B020 = 0xB020,
	LOG_POINT_B021 = 0xB021,

	LOG_POINT_B030 = 0xB030,
	LOG_POINT_B031 = 0xB031,
	LOG_POINT_B032 = 0xB032,
	LOG_POINT_B033 = 0xB033,
	LOG_POINT_B034 = 0xB034,
	LOG_POINT_B035 = 0xB035,
	LOG_POINT_B036 = 0xB036,
	LOG_POINT_B037 = 0xB037,

	LOG_POINT_B040 = 0xB040,
	LOG_POINT_B041 = 0xB041,
	LOG_POINT_B042 = 0xB042,
	LOG_POINT_B043 = 0xB043,
	LOG_POINT_B044 = 0xB044,

	LOG_POINT_B050 = 0xB050,
	LOG_POINT_B051 = 0xB051,
	LOG_POINT_B052 = 0xB052,
	LOG_POINT_B053 = 0xB053,
	LOG_POINT_B054 = 0xB054,
	LOG_POINT_B055 = 0xB055,

	LOG_POINT_B05A = 0xB05A,
	LOG_POINT_B05B = 0xB05B,
	LOG_POINT_B05C = 0xB05C,
	LOG_POINT_B05D = 0xB05D,

	LOG_POINT_B060 = 0xB060,
	LOG_POINT_B061 = 0xB061,

	LOG_POINT_B070 = 0xB070,
	LOG_POINT_B071 = 0xB071,
	LOG_POINT_B072 = 0xB072,
	LOG_POINT_B073 = 0xB073,
	LOG_POINT_B074 = 0xB074,
	LOG_POINT_B075 = 0xB075,

	LOG_POINT_B080 = 0xB080,
	LOG_POINT_B081 = 0xB081,
	LOG_POINT_B082 = 0xB082,
	LOG_POINT_B083 = 0xB083,
	LOG_POINT_B084 = 0xB084,
	LOG_POINT_B085 = 0xB085,
	LOG_POINT_B086 = 0xB086,
	LOG_POINT_B087 = 0xB087,
	LOG_POINT_B088 = 0xB088,
	LOG_POINT_B089 = 0xB089,
	LOG_POINT_B08A = 0xB08A,
	LOG_POINT_B08F = 0xB08F,

	LOG_POINT_B090 = 0xB090,
	LOG_POINT_B091 = 0xB091,
	LOG_POINT_B092 = 0xB092,
	LOG_POINT_B093 = 0xB093,

	LOG_POINT_B0A0 = 0xB0A0,
	LOG_POINT_B0A1 = 0xB0A1,
	LOG_POINT_B0A2 = 0xB0A2,
	LOG_POINT_B0A3 = 0xB0A3,
	LOG_POINT_B0A4 = 0xB0A4,
	LOG_POINT_B0A5 = 0xB0A5,

	LOG_POINT_B0B0 = 0xB0B0,
	LOG_POINT_B0B1 = 0xB0B1,
	LOG_POINT_B0B2 = 0xB0B2,
	LOG_POINT_B0B3 = 0xB0B3,
	LOG_POINT_B0B4 = 0xB0B4,
	LOG_POINT_B0B5 = 0xB0B5,
	LOG_POINT_B0B6 = 0xB0B6,
	LOG_POINT_B0B7 = 0xB0B7,
	LOG_POINT_B0B8 = 0xB0B8,
	LOG_POINT_B0B9 = 0xB0B9,
	LOG_POINT_B0BF = 0xB0BF,


	LOG_POINT_B100 = 0xB100,
	LOG_POINT_B101 = 0xB101,
	LOG_POINT_B102 = 0xB102,
	LOG_POINT_B103 = 0xB103,
	LOG_POINT_B104 = 0xB104,
	LOG_POINT_B105 = 0xB105,
	LOG_POINT_B106 = 0xB106,
	LOG_POINT_B107 = 0xB107,
	LOG_POINT_B108 = 0xB108,
	// Key Remap
	LOG_POINT_C000 = 0xC000,
	LOG_POINT_C001 = 0xC001,
	LOG_POINT_C002 = 0xC002,
	LOG_POINT_C003 = 0xC003,
	LOG_POINT_C004 = 0xC004,
	LOG_POINT_C005 = 0xC005,
	LOG_POINT_C006 = 0xC006,

	LOG_POINT_C010 = 0xC010,

	//Led Remap
	LOG_POINT_C100 = 0xC100,
	LOG_POINT_C101 = 0xC101,
	LOG_POINT_C102 = 0xC102,
	LOG_POINT_C103 = 0xC103,
	//battery
	LOG_POINT_C200 = 0xC200,
	LOG_POINT_C201 = 0xC201,
	LOG_POINT_C202 = 0xC202,
	LOG_POINT_C203 = 0xC203,
	LOG_POINT_C204 = 0xC204,
	LOG_POINT_C205 = 0xC205,
	LOG_POINT_C210 = 0xC210,
	LOG_POINT_C211 = 0xC211,
	LOG_POINT_C212 = 0xC212,
	LOG_POINT_C213 = 0xC213,
	LOG_POINT_C214 = 0xC214,
	LOG_POINT_C215 = 0xC215,
	LOG_POINT_C216 = 0xC216,
	LOG_POINT_C220 = 0xC220,
	LOG_POINT_C221 = 0xC221,
	LOG_POINT_C222 = 0xC222,
	LOG_POINT_C223 = 0xC223,
	LOG_POINT_C224 = 0xC224,
	//ATCMD
	LOG_POINT_C300=0xC300,
	LOG_POINT_C301=0xC301,
	//BLE OTA
	LOG_POINT_C400=0xC400,
	LOG_POINT_C401=0xC401,
	LOG_POINT_C402=0xC402,
	//OAL
	LOG_POINT_C500=0xC500,
	LOG_POINT_C501=0xC501,
	LOG_POINT_C502=0xC502,
	LOG_POINT_C503=0xC503,
	LOG_POINT_C504=0xC504,
	LOG_POINT_C505=0xC505,
	LOG_POINT_C506=0xC506,
	// SCO
	LOG_POINT_D000 = 0xD000,
	LOG_POINT_D001 = 0xD001,
	LOG_POINT_D002 = 0xD002,
	LOG_POINT_D003 = 0xD003,
	LOG_POINT_D004 = 0xD004,
	LOG_POINT_D005 = 0xD005,
	LOG_POINT_D006 = 0xD006,
	LOG_POINT_D007 = 0xD007,
	LOG_POINT_D008 = 0xD008,
	LOG_POINT_D009 = 0xD009,
	LOG_POINT_D00A = 0xD00A,
	LOG_POINT_D00B = 0xD00B,
	LOG_POINT_D00C = 0xD00C,
	LOG_POINT_D00D = 0xD00D,
	LOG_POINT_D00E = 0xD00E,
	LOG_POINT_D00F = 0xD00F,

	// CVSD
	LOG_POINT_D010 = 0xD010,
	LOG_POINT_D011 = 0xD011,
	LOG_POINT_D012 = 0xD012,
	LOG_POINT_D013 = 0xD013,
	LOG_POINT_D014 = 0xD014,
	LOG_POINT_D015 = 0xD015,
	LOG_POINT_D016 = 0xD016,
	LOG_POINT_D100 = 0xD100,
	LOG_POINT_D101 = 0xD101,
	LOG_POINT_D102 = 0xD102,
	LOG_POINT_D103 = 0xD103,
	LOG_POINT_D104 = 0xD104,
	LOG_POINT_D105 = 0xD105,
	LOG_POINT_D106 = 0xD106,
	LOG_POINT_D107 = 0xD107,
	LOG_POINT_D108 = 0xD108,
	LOG_POINT_D109 = 0xD109,
	LOG_POINT_D10a = 0xD10a,
	LOG_POINT_D10b = 0xD10b,

	// HCI
	// hci snd.
	// evt
	LOG_POINT_E000 = 0xE000,
	LOG_POINT_E001 = 0xE001,
	LOG_POINT_E002 = 0xE002,
	LOG_POINT_E003 = 0xE003,
	LOG_POINT_E004 = 0xE004,
	LOG_POINT_E005 = 0xE005,

	// acl
	LOG_POINT_E010 = 0xE010,
	LOG_POINT_E011 = 0xE011,
	// hci rcv.
	// cmd
	LOG_POINT_E100 = 0xE100,
	LOG_POINT_E101 = 0xE101,
	// acl
	LOG_POINT_E110 = 0xE110,
	LOG_POINT_E111 = 0xE111,
	// For Error Debug, from 0xfff9 to 0xfffe use for cm0
	LOG_POINT_FFF9 = 0xFFF9,
};

#endif